--[[FMPM package metadata:
info = "Image library"
location = "/lib/image.lua"
]]

-- img = image.loadFromFile(fname)
-- img = image.loadFromString(data)

-- img:getColorAt(x,y)
-- img:getColorId(x,y)
-- img:getColorCount()
-- img:getColorPalette()
-- img:getDimensions()
-- img:getSize()
-- img:getType()

-- img:drawInline()
-- img:drawFullscreen()
-- img:drawAt(x,y)

-- Not yet:
-- createNew(sizeX,sizeY,colors)
-- setColorPalette()
-- setDimensions()
-- setColorCount()
-- saveToString()

local image = {}
local component = require("component")
local term = require("term")
local gpu = component.gpu

function rgbColor(r, g, b)
	return b + (g*256) + (r*65536)
end

function leValue(data, offset, bytes)
	value = 0
	for cb = bytes - 1, 0, -1 do
		value = value * 256 + data:byte(offset + cb)
	end
	return value
end

function image.loadFromFile(fname)
	local file = io.open(fname, "rb")
	if not file then
		error("Unable to access " .. fname .. ".")
	end
	local data = file:read("*a")
	return image.loadFromString(data)
end

function image.loadFromString(data)
	local self = setmetatable({}, {__index = image})
	if data:sub(1,2) == "BM" then
		local headerLength = leValue(data, 15, 4)
		if headerLength == 40 then
			self.data = data
			self.type = "BMP"
			self.size = leValue(data, 3, 4)
			self.width = leValue(data, 19, 4)
			self.height = leValue(data, 23, 4)
			self.colorCount = leValue(data, 47, 4)
			self.paletteOffset = 15 + headerLength
			self.pixelOffset = leValue(data, 11, 4) + 1
			self.palette = {}
			for col = 0, self.colorCount - 1, 1 do
				self.colorOffset = self.paletteOffset + (4*col)
				b, g, r = leValue(self.data, self.colorOffset, 1), leValue(self.data, self.colorOffset + 1, 1), leValue(self.data, self.colorOffset + 2, 1)
				self.palette[col+1] = rgbColor(r, g, b)
			end
			return self
		else
			error("This BMP header is not supported.")
		end
	else
		error("Image format not supported.")
	end
end

function image.drawInline(self)
	local cursorX, cursorY = term.getCursor()
	padding = (self.width % 4)
	for y = self.height-1, 0, -2 do
		term.setCursor(cursorX, cursorY)
		for x = 0, self.width-1, 1 do
			current = self.pixelOffset + x
			gpu.setForeground(self.palette[leValue(self.data, current + (y*self.width) + padding*y, 1) + 1])
			gpu.setBackground(self.palette[leValue(self.data, current + ((y-1)*self.width) + padding*(y-1), 1) + 1])
			io.write("▀")
		end
		cursorY = cursorY + 1
	end
	gpu.setForeground(0xFFFFFF)
	gpu.setBackground(0x000000)
end

function image.drawAt(self,x,y)
	term.setCursor(x,y)
	self:drawInline()
end

function image.drawFullscreen(self)
	gpu.setResolution(self.width, (self.height/2) + 2)
	self:drawAt(1,1)
end

function image.getDataString(self)
	return self.data
end

function image.getType(self)
	return self.type
end

function image.getSize(self)
	return self.size
end

function image.getWidth(self)
	return self.width
end

function image.getHeight(self)
	return self.height
end

function image.getColorCount(self)
	return self.colorCount
end

return image